import {aiAgentQueries} from '@ai/ai-agent/ai-agent-queries';
import {AiAgentSettings} from '@ai/ai-agent/settings/ai-agent-settings';
import {PanelLayout} from '@ai/ai-agent/settings/panel-layout';
import {useSuspenseQuery} from '@tanstack/react-query';
import {AccordionItemProps} from '@ui/accordion/accordion';
import {FormChipField} from '@ui/forms/input-field/chip-field/form-chip-field';
import {FormTextField} from '@ui/forms/input-field/text-field/text-field';
import {Item} from '@ui/forms/listbox/item';
import {FormRadio} from '@ui/forms/radio-group/radio';
import {FormSelect} from '@ui/forms/select/select';
import {Trans} from '@ui/i18n/trans';
import {createSvgIcon} from '@ui/icons/create-svg-icon';
import {useForm, useWatch} from 'react-hook-form';

export function GreetingPanel(props: Partial<AccordionItemProps>) {
  const {data} = useSuspenseQuery(aiAgentQueries.settings.index());
  const form = useForm<Partial<AiAgentSettings>>({
    defaultValues: {
      greetingType: data.settings.greetingType ?? 'basicGreeting',
      initialFlowId: data.settings.initialFlowId ?? data.flows[0]?.id ?? '',
      basicGreeting: {
        message: data.settings.basicGreeting?.message ?? '',
        flowIds: data.settings.basicGreeting?.flowIds ?? [],
      },
    },
  });
  const greetingType = useWatch({control: form.control, name: 'greetingType'});

  return (
    <PanelLayout
      {...props}
      label={<Trans message="Start of the conversation" />}
      description={
        <Trans message="Show a greeting and buttons or start with a flow" />
      }
      icon={<WavingHandIcon />}
      form={form}
    >
      <div className="mb-28">
        <FormRadio name="greetingType" value="basicGreeting" size="sm">
          <Trans message="Send a greeting message and show buttons" />
        </FormRadio>
        {greetingType === 'basicGreeting' && (
          <div className="pl-24 pt-16">
            <FormTextField
              name="basicGreeting.message"
              label={<Trans message="Message" />}
              inputElementType="textarea"
              rows={4}
              className="mb-16"
            />
            <FormChipField
              name="basicGreeting.flowIds"
              label={<Trans message="Flows (up to 10)" />}
              allowCustomValue={false}
              maxItems={10}
              valueKey="id"
              displayWith={value =>
                data.flows.find(flow => flow.id === value.id)?.name ?? ''
              }
              suggestions={data.flows}
            >
              {flow => (
                <Item key={flow.id} value={flow.id}>
                  {flow.name}
                </Item>
              )}
            </FormChipField>
          </div>
        )}
      </div>
      <div>
        <FormRadio name="greetingType" value="flow" size="sm">
          <Trans message="Start with a specific flow" />
        </FormRadio>
        {greetingType === 'flow' && (
          <div className="pl-24 pt-16">
            <FormSelect
              name="initialFlowId"
              selectionMode="single"
              label={<Trans message="Select a flow" />}
              required
            >
              {data.flows.map(flow => (
                <Item key={flow.id} value={flow.id}>
                  {flow.name}
                </Item>
              ))}
            </FormSelect>
          </div>
        )}
      </div>
    </PanelLayout>
  );
}

const WavingHandIcon = createSvgIcon(
  <path d="M7.03,4.95L3.49,8.49c-3.32,3.32-3.32,8.7,0,12.02s8.7,3.32,12.02,0l6.01-6.01c0.97-0.97,0.97-2.56,0-3.54 c-0.12-0.12-0.25-0.23-0.39-0.32l0.39-0.39c0.97-0.97,0.97-2.56,0-3.54c-0.16-0.16-0.35-0.3-0.54-0.41c0.4-0.92,0.23-2.02-0.52-2.77 c-0.87-0.87-2.22-0.96-3.2-0.28c-0.1-0.15-0.21-0.29-0.34-0.42c-0.97-0.97-2.56-0.97-3.54,0l-2.51,2.51 c-0.09-0.14-0.2-0.27-0.32-0.39C9.58,3.98,8,3.98,7.03,4.95z M8.44,6.37c0.2-0.2,0.51-0.2,0.71,0c0.2,0.2,0.2,0.51,0,0.71 l-3.18,3.18c1.17,1.17,1.17,3.07,0,4.24l1.41,1.41c1.45-1.45,1.82-3.57,1.12-5.36l6.3-6.3c0.2-0.2,0.51-0.2,0.71,0s0.2,0.51,0,0.71 l-4.6,4.6l1.41,1.41l6.01-6.01c0.2-0.2,0.51-0.2,0.71,0c0.2,0.2,0.2,0.51,0,0.71l-6.01,6.01l1.41,1.41l4.95-4.95 c0.2-0.2,0.51-0.2,0.71,0c0.2,0.2,0.2,0.51,0,0.71l-5.66,5.66l1.41,1.41l3.54-3.54c0.2-0.2,0.51-0.2,0.71,0c0.2,0.2,0.2,0.51,0,0.71 L14.1,19.1c-2.54,2.54-6.65,2.54-9.19,0s-2.54-6.65,0-9.19L8.44,6.37z M23,17c0,3.31-2.69,6-6,6v-1.5c2.48,0,4.5-2.02,4.5-4.5H23z M1,7c0-3.31,2.69-6,6-6v1.5C4.52,2.5,2.5,4.52,2.5,7H1z" />,
  'WavingHand',
);
