<?php

namespace Common\Files\Controllers;

use Common\Core\BaseController;
use Common\Files\UploadDiskResolver;
use Exception;
use Google\Service\AndroidEnterprise\Install;
use Illuminate\Support\Arr;
use League\Flysystem\AwsS3V3\AwsS3V3Adapter;
use League\Flysystem\Ftp\FtpAdapter;
use Spatie\FlysystemDropbox\DropboxAdapter;

class ValidateBackendCredentialsController extends BaseController
{
    public function __construct()
    {
        $this->middleware('isAdmin');
    }

    public function __invoke()
    {
        $this->blockOnDemoSite();

        // validate payload
        $data = $this->validate(request(), [
            'name' => ['string', 'required'],
            'type' => ['string', 'required'],
            'root' => ['string', 'nullable'],
            'credentials' => ['array', 'required'],
        ]);

        // check if credentials are valid by trying to list files on actual disk
        try {
            $disk = (new UploadDiskResolver())->resolveFromConfig($data, [
                'visibility' => 'private',
                'root' => '',
            ]);

            // todo: test dropbox
            if ($disk->getAdapter() instanceof DropboxAdapter) {
                // dropbox adapter catches all errors silently
                // need to use client directly to check for errors
                $disk->getAdapter()->getClient()->listFolder();
            } else {
                $disk->allFiles();
            }
        } catch (Exception $e) {
            dd($e);
            abort(
                422,
                __(
                    'These credentials are invalid. Please double-check them and try again.',
                ),
            );
        }

        // if s3 and direct upload is enabled, configure cors
        if (
            $disk->getAdapter() instanceof AwsS3V3Adapter &&
            Arr::get($data, 'credentials.direct_upload')
        ) {
            $cors = [
                [
                    'AllowedOrigins' => [config('app.url')],
                    'AllowedMethods' => ['GET', 'HEAD', 'POST', 'PUT'],
                    'MaxAgeSeconds' => 3000,
                    'AllowedHeaders' => ['*'],
                    'ExposeHeaders' => ['ETag'],
                ],
            ];

            try {
                $disk->getClient()->putBucketCors([
                    'Bucket' => $data['credentials']['bucket'],
                    'CORSConfiguration' => [
                        'CORSRules' => $cors,
                    ],
                ]);
            } catch (Exception $e) {
                abort(
                    422,
                    __(
                        'Could not configure bucket for direct upload. Make sure bucket exists and you have required permissions.',
                    ),
                );
            }
        }

        return $this->success();
    }
}
