<?php namespace Common\Core\Bootstrap;

use App\Models\User;
use Common\Auth\Roles\Role;
use Common\Billing\Gateways\Stripe\FormatsMoney;
use Common\Core\AppUrl;
use Common\Files\UploadDiskResolver;
use Common\Localizations\Localization;
use Common\Localizations\LocalizationsRepository;
use Common\Settings\Settings;
use Common\Settings\Themes\CssTheme;
use Common\Websockets\GetWebsocketCredentialsForClient;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\View;
use Jenssegers\Agent\Agent;

class BaseBootstrapData implements BootstrapData
{
    use FormatsMoney;

    protected array $data = [];
    public ?CssTheme $initialTheme = null;

    public function __construct(
        protected Settings $settings,
        protected Request $request,
        protected Role $role,
        protected LocalizationsRepository $localizationsRepository,
    ) {}

    public function getEncoded(): string
    {
        if ($this->data['user']) {
            $this->data['user'] = $this->data['user']->toArray();
        }

        return json_encode($this->data);
    }

    public function set(string $key, mixed $value): static
    {
        Arr::set($this->data, $key, $value);
        return $this;
    }

    public function get($key = null)
    {
        return $key ? Arr::get($this->data, $key) : $this->data;
    }

    public function init(): self
    {
        $this->data['settings'] = settings()->getUnflattened();
        $this->data['settings'][
            'broadcasting'
        ] = (new GetWebsocketCredentialsForClient())->execute();
        $this->data['csrf_token'] = csrf_token();
        $this->data['is_mobile_device'] = app(Agent::class)->isMobile();
        $this->data['settings']['base_url'] = config('app.url');
        $this->data['settings']['asset_url'] = config('app.asset_url');
        $this->data['settings']['html_base_uri'] = app(
            AppUrl::class,
        )->htmlBaseUri;
        $this->data['settings']['version'] = config('app.version');
        $this->data['sentry_release'] = config('sentry.release');
        $this->data['default_meta_tags'] = $this->getDefaultMetaTags();
        $this->data['user'] = $this->getCurrentUser();
        $this->data['guest_role'] = app('guestRole')?->load('permissions');
        $this->data['themes'] = $this->getThemes();
        $this->setInitialTheme();
        $this->setUploadingTypes();

        $this->data['i18n'] = [
            'locales' => Localization::query()
                ->select(['name', 'language'])
                ->get()
                ->map(function (Localization $localization) {
                    if (
                        settings('i18n.enable', true) &&
                        $localization->language === app()->getLocale()
                    ) {
                        $localization->loadLines();
                    }
                    return [
                        'name' => $localization->name,
                        'language' => $localization->language,
                        'lines' => $localization->lines ?? [],
                    ];
                }),
            'active' => app()->getLocale(),
        ];

        if (config('app.notifications_integrated') && $this->data['user']) {
            $this->data['user']->loadCount('unreadNotifications');
        }

        $alreadyAccepted =
            !settings('cookie_notice.enable') ||
            (bool) Arr::get($_COOKIE, 'cookie_notice', false);
        $this->data['show_cookie_notice'] =
            !$alreadyAccepted && $this->isCookieLawCountry();

        $this->data['is_settings_preview'] =
            request('settingsPreview') === 'true';

        if ($this->data['user']) {
            $this->data['user']->createOrTouchSession();
        }

        return $this;
    }

    protected function setInitialTheme(): void
    {
        $themes = $this->data['themes'];

        // first, get theme from cookie
        if (
            $cookieTheme = $themes->find(Arr::get($_COOKIE, 'be-active-theme'))
        ) {
            $this->initialTheme = $themes->find($cookieTheme);
        }

        // if no theme was selected, get default theme specified by admin
        elseif ($defaultId = settings('themes.default_id')) {
            // when default theme is set to system, use light theme
            // initially as there's no way to get user's preference
            // without javascript. Correct theme variables will be set once front end loads.
            if ($defaultId === 'system' || $defaultId == 0) {
                $this->initialTheme = $themes
                    ->where('default_light', true)
                    ->first();
            } else {
                $this->initialTheme = $themes->find($defaultId);
            }
        }

        // finally, fallback to default light theme
        if (!$this->initialTheme) {
            $this->initialTheme =
                $themes->where('default_light', true)->first() ??
                $themes->first();
        }
    }

    public function getThemes(): Collection
    {
        $themes = CssTheme::where('type', 'site')
            ->where(function (Builder $builder) {
                $builder
                    ->where('default_dark', true)
                    ->orWhere('default_light', true);
            })
            ->get();

        if ($themes->isEmpty()) {
            $themes = CssTheme::limit(2)->get();
        }

        return $themes;
    }

    /**
     * Load current user and his roles.
     */
    public function getCurrentUser(): ?User
    {
        $user = $this->request->user();
        if ($user) {
            // load user subscriptions, if billing is enabled
            if (
                settings('billing.enable') &&
                !$user->relationLoaded('subscriptions')
            ) {
                $user->load('subscriptions.price');
            }

            // load user roles, if not already loaded
            if (!$user->relationLoaded('roles')) {
                $user->load('roles');
            }

            if (!$user->relationLoaded('permissions')) {
                $user->loadPermissions();
            }
        }

        return $user;
    }

    protected function setUploadingTypes()
    {
        if (!config('app.use_new_uploading')) {
            return;
        }

        $config = UploadDiskResolver::getAllUploadTypesConfig();
        $configForFrontend = [];

        foreach ($config as $uploadType => $config) {
            $configForFrontend[$uploadType] = [
                'backends' => array_map(function ($backendId) {
                    $config = UploadDiskResolver::getBackendConfig($backendId);
                    $type = str_ends_with($config['type'], 's3')
                        ? 's3'
                        : $config['type'];
                    $data = [
                        'id' => $backendId,
                        'type' => $type,
                    ];

                    if ($type === 's3') {
                        $data['direct_upload'] =
                            $config['credentials']['direct_upload'] ?? false;
                    }

                    return $data;
                }, $config['backends']),
                'visibility' => $config['visibility'],
            ];
        }

        // not needed for frontend
        unset($this->data['uploading']['backends']);
        unset($this->data['uploading']['types']);

        // set only partial config for frontend
        $this->data['uploading_types'] = $configForFrontend;
    }

    protected function getDefaultMetaTags(): string
    {
        $pageName = 'landing-page';
        if (View::exists("editable-views::seo-tags.$pageName")) {
            return view("editable-views::seo-tags.$pageName")->render();
        } else {
            return view("seo.$pageName.seo-tags")->render();
        }
    }

    protected function isCookieLawCountry(): bool
    {
        $isoCode = geoip(getIp())['iso_code'];
        // prettier-ignore
        return in_array($isoCode, ['AT', 'BE', 'BG', 'BR', 'CY', 'CZ', 'DE', 'DK', 'EE', 'EL', 'ES', 'FI', 'FR', 'GB', 'HR', 'HU', 'IE', 'IT','LT', 'LU', 'LV', 'MT', 'NL', 'NO', 'PL', 'PT', 'RO', 'SE', 'SI', 'SK',
        ]);
    }
}
