import {BackendResponse} from '@common/http/backend-response/backend-response';
import {getAxiosErrorMessage} from '@common/http/get-axios-error-message';
import {apiClient} from '@common/http/query-client';
import {showHttpErrorToast} from '@common/http/show-http-error-toast';
import {useMutation} from '@tanstack/react-query';
import {message} from '@ui/i18n/message';
import {toast} from '@ui/toast/toast';
import {User} from '@ui/types/user';
import {UploadedFile} from '@ui/utils/files/uploaded-file';

interface Response extends BackendResponse {
  user: User;
}

interface Payload {
  file?: UploadedFile;
  url?: string;
}

interface Props {
  user: {id: number};
}
export function useUploadAvatar({user}: Props) {
  return useMutation({
    mutationFn: (payload: Payload) => UploadAvatar(payload, user),
    onSuccess: () => {
      toast(message('Uploaded avatar'));
    },
    onError: err => {
      const message = getAxiosErrorMessage(err, 'file');
      if (message) {
        toast.danger(message);
      } else {
        showHttpErrorToast(err);
      }
    },
  });
}

function UploadAvatar(
  {file, url}: Payload,
  user: {id: number},
): Promise<Response> {
  const payload = new FormData();
  if (file) {
    payload.set('file', file.native);
  } else {
    payload.set('url', url!);
  }
  return apiClient
    .post(`users/${user.id}/avatar`, payload, {
      headers: {
        'Content-Type': 'multipart/form-data',
      },
    })
    .then(r => r.data);
}
