import {useUpdateAdminSettings} from '@common/admin/settings/requests/use-update-admin-settings';
import {useAdminSettingsPageNavConfig} from '@common/admin/settings/use-admin-settings-page-nav-config';
import {DatatablePageHeaderBar} from '@common/datatable/page/datatable-page-with-header-layout';
import {useNavigate} from '@common/ui/navigation/use-navigate';
import {Button} from '@ui/buttons/button';
import {IconButton} from '@ui/buttons/icon-button';
import {Form} from '@ui/forms/form';
import {Item} from '@ui/forms/listbox/item';
import {MessageDescriptor} from '@ui/i18n/message-descriptor';
import {Trans} from '@ui/i18n/trans';
import {MenuIcon} from '@ui/icons/material/Menu';
import {Menu, MenuTrigger} from '@ui/menu/menu-trigger';
import {BlockerDialog} from '@ui/overlays/dialog/blocker-dialog';
import {ProgressCircle} from '@ui/progress/progress-circle';
import {useIsMobileMediaQuery} from '@ui/utils/hooks/is-mobile-media-query';
import {isAbsoluteUrl} from '@ui/utils/urls/is-absolute-url';
import {Fragment, ReactElement, ReactNode} from 'react';
import {FieldErrors, UseFormReturn, useFormState} from 'react-hook-form';
import {useLocation} from 'react-router';

interface Props {
  title: ReactElement<MessageDescriptor>;
  form: UseFormReturn<any>;
  isPending: boolean;
}
export function SettingsPageHeader({title, form, isPending}: Props) {
  const isMobile = useIsMobileMediaQuery();
  const {isDirty} = useFormState({control: form.control});

  const submitButton = (
    <Button
      type="submit"
      variant="flat"
      color="primary"
      size="xs"
      startIcon={
        isPending ? <ProgressCircle size="xs" isIndeterminate /> : null
      }
      disabled={isPending || !isDirty}
    >
      <Trans message="Save changes" />
    </Button>
  );

  return (
    <Fragment>
      <DatatablePageHeaderBar
        title={title}
        showSidebarToggleButton={!!isMobile}
        rightContent={
          <Fragment>
            {isMobile && <MobileNav />}
            {submitButton}
          </Fragment>
        }
      />
      <BlockerDialog shouldBlock={isDirty} />
    </Fragment>
  );
}

interface SettingsFormProps {
  id: string;
  form: UseFormReturn<any>;
  children: ReactNode;
}
export function SettingsForm({id, form, children}: SettingsFormProps) {
  const updateSettings = useUpdateAdminSettings(form);
  return (
    <Form
      id={id}
      form={form}
      onBeforeSubmit={() => {
        // clear group errors, because hook form won't automatically
        // clear errors that are not bound to a specific form field
        const errors = form.formState.errors as FieldErrors<object>;
        const keys = Object.keys(errors).filter(key => {
          return key.endsWith('_group');
        });
        form.clearErrors(keys as any);
      }}
      onSubmit={values => {
        updateSettings.mutate(values, {
          onSuccess: () => form.reset(values),
        });
      }}
    >
      {children}
    </Form>
  );
}

function MobileNav() {
  const {pathname} = useLocation();
  const navigate = useNavigate();
  const value = pathname.split('/').pop();

  const navConfig = useAdminSettingsPageNavConfig();

  return (
    <MenuTrigger
      selectionMode="single"
      selectedValue={value}
      onSelectionChange={newPage => {
        newPage = !isAbsoluteUrl(newPage as string)
          ? `/admin/settings/${newPage}`
          : newPage;
        navigate(newPage as string, {state: {prevPath: pathname}});
      }}
    >
      <IconButton>
        <MenuIcon />
      </IconButton>
      <Menu>
        {navConfig.map(item => (
          <Item key={item.to as string} value={item.to}>
            <Trans {...item.label} />
          </Item>
        ))}
      </Menu>
    </MenuTrigger>
  );
}
