<?php

namespace App\HelpCenter\Actions;

use App\HelpCenter\Models\HcArticle;
use App\HelpCenter\Models\HcCategory;
use Common\Files\FileEntry;
use Common\Search\ImportRecordsIntoScout;
use Common\Tags\Tag;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use ZipArchive;

class ImportHelpCenter
{
    public function execute(string $path): void
    {
        $zip = new ZipArchive();
        $zip->open($path);

        // store help center images
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $stat = $zip->statIndex($i);
            if (Str::startsWith(dirname($stat['name']), 'images/')) {
                [$root, $folder, $name] = explode('/', $stat['name']);
                Storage::disk('public')->put(
                    "$folder/$name",
                    $zip->getFromIndex($stat['index']),
                );
            }
        }

        $helpCenter = json_decode($zip->getFromName('help-center.json'), true);
        $oldBaseUri = $zip->getFromName('base-url.txt');
        $zip->close();

        // delete old help center data and truncate tables
        (new DeleteMultipleArticles())->execute(
            DB::table('articles')->pluck('id')->toArray(),
        );
        DB::table('articles')->truncate();
        DB::table('categories')->truncate();
        DB::table('category_article')->truncate();

        HcArticle::disableSearchSyncing();

        try {
            Artisan::call('scout:flush ' . HcArticle::class);
        } catch (\Exception) {
        }

        // unguard models
        HcCategory::unguard();
        HcArticle::unguard();
        Tag::unguard();
        FileEntry::unguard();

        foreach ($helpCenter as $categoryData) {
            // create categories
            $category = HcCategory::create([
                ...Arr::except($categoryData, [
                    'sections',
                    'id',
                    'model_type',
                    'is_section',
                    'hidden',
                ]),
                'created_at' => $categoryData['created_at'] ?? now(),
                'updated_at' => $categoryData['updated_at'] ?? now(),
            ]);

            // create sections
            collect($categoryData['sections'])->each(function (
                $sectionData,
            ) use ($category, $oldBaseUri) {
                $sectionData['parent_id'] = $category->id;
                $section = HcCategory::create([
                    ...Arr::except($sectionData, [
                        'articles',
                        'id',
                        'model_type',
                        'is_section',
                        'hidden',
                    ]),
                    'created_at' => $sectionData['created_at'] ?? now(),
                    'updated_at' => $sectionData['updated_at'] ?? now(),
                ]);

                // create articles
                foreach ($sectionData['articles'] as $articleData) {
                    $articleData[
                        'body'
                    ] = (new ReplaceArticleBodyUrls())->replaceImgSrc(
                        $articleData['body'],
                        [$oldBaseUri],
                    );

                    $article = HcArticle::firstOrCreate([
                        ...Arr::except($articleData, [
                            'tags',
                            'id',
                            'model_type',
                            'article_id',
                            'category_id',
                            'position',
                            'extra_data',
                        ]),
                        'created_at' => $articleData['created_at'] ?? now(),
                        'updated_at' => $articleData['updated_at'] ?? now(),
                        'author_id' => Arr::get($articleData, 'author_id')
                            ? $articleData['author_id']
                            : Auth::id(),
                    ]);
                    $article
                        ->categories()
                        ->attach([$section->id, $category->id]);

                    // create tags
                    foreach ($articleData['tags'] as $tagData) {
                        $attrs = [
                            'name' => $tagData['name'],
                        ];
                        $tag = Tag::where('name', $tagData['name'])->first();
                        if (!$tag) {
                            $tag = Tag::create($attrs);
                        }

                        try {
                            $article->tags()->attach($tag->id);
                        } catch (\Exception) {
                        }
                    }
                }
            });
        }

        Cache::flush();
        (new ImportRecordsIntoScout())->execute(HcArticle::class);
    }
}
